<?php

namespace WP_Statistics\Service\Admin\PrivacyAudit\Audits;

use WP_Statistics\Components\View;
use WP_STATISTICS\Option;
use WP_Statistics\Service\Admin\PrivacyAudit\Audits\Abstracts\RecommendedAudit;
use WP_Statistics\Service\Admin\PrivacyAudit\Faqs\RequireConsent;

class RecommendConsent extends RecommendedAudit
{
    public static function getStatus()
    {
        $consentLevel  = Option::get('consent_level_integration');
        $consentStatus = RequireConsent::getStatus();

        if (!empty($consentLevel) && $consentLevel != 'disabled') {
            return 'passed';
        }

        return ($consentStatus === 'warning') ? 'action_required' : 'recommended';
    }

    public static function getPassedStateInfo()
    {
        return [
            'icon'  => '<svg width="28" height="28" viewBox="0 0 28 28" fill="none" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink"><rect x="0.5" width="27" height="28" fill="url(#pattern0_19258_212788)"></rect><defs><pattern id="pattern0_19258_212788" patternContentUnits="objectBoundingBox" width="1" height="1"><use xlink:href="#image0_19258_212788" transform="matrix(0.00566687 0 0 0.00546448 -0.00435135 0)"></use></pattern><image id="image0_19258_212788" width="178" height="183" xlink:href="data:image/png;base64,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"></image></defs></svg>',
            'title' => esc_html__('Consent Plugin Successfully Integrated', 'wp-statistics'),
            'notes' => View::load('components/privacy-audit/recommend-consent', [], true),
        ];
    }

    public static function getUnpassedStateInfo()
    {
        return [
            'icon'  => '<svg width="28" height="28" viewBox="0 0 28 28" fill="none" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink"><rect x="0.5" width="27" height="28" fill="url(#pattern0_19258_212788)"></rect><defs><pattern id="pattern0_19258_212788" patternContentUnits="objectBoundingBox" width="1" height="1"><use xlink:href="#image0_19258_212788" transform="matrix(0.00566687 0 0 0.00546448 -0.00435135 0)"></use></pattern><image id="image0_19258_212788" width="178" height="183" xlink:href="data:image/png;base64,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"></image></defs></svg>',
            'title' => esc_html__('Consent Plugin Required for Privacy Compliance', 'wp-statistics'),
            'notes' => View::load('components/privacy-audit/recommend-consent-unpassed', [], true),
        ];
    }

    public static function getRecommendedStateInfo()
    {
        return [
            'icon'  => '<svg width="28" height="28" viewBox="0 0 28 28" fill="none" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink"><rect x="0.5" width="27" height="28" fill="url(#pattern0_19258_212788)"></rect><defs><pattern id="pattern0_19258_212788" patternContentUnits="objectBoundingBox" width="1" height="1"><use xlink:href="#image0_19258_212788" transform="matrix(0.00566687 0 0 0.00546448 -0.00435135 0)"></use></pattern><image id="image0_19258_212788" width="178" height="183" xlink:href="data:image/png;base64,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"></image></defs></svg>',
            'title' => esc_html__('Respect Visitor Privacy with Consent Plugins', 'wp-statistics'),
            'notes' => View::load('components/privacy-audit/recommend-consent-info', [], true),
        ];
    }
}